<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use App\Models\User;
use App\Models\Role;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Facades\Validator;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Http;
use Carbon\Carbon;

class UserController extends Controller
{
    public $user;

    public function __construct()
    {
        $this->middleware(function ($request, $next) {
            $this->user = Auth::guard('web')->user();
            return $next($request);
        });
    }

    public function index()
    {
        if (!auth()->user()->can('View User')) {
            abort(403, 'Sorry !! You are not authorized!');
        }

        $users = User::all();
        return view('user.index', compact('users'));
    }

    public function create()
    {
        if (!auth()->user()->can('Create User')) {
            abort(403, 'Sorry !! You are not authorized!');
        }

        $roles = Role::all();
        return view('user.create', compact('roles'));
    }

    public function store(Request $request)
    {
        if (!auth()->user()->can('Create User')) {
            abort(403, 'Sorry !! You are not authorized!');
        }

        $validator = Validator::make($request->all(), [
            'name' => 'required',
            'email' => 'required|email|unique:users',
            'password' => 'required',
        ]);

        if ($validator->fails()) {
            return redirect()->route('users.create')
                ->with('error', 'Missing required fields.')->withInput()->withErrors($validator);
        }

        $token = session('api_token');
        $api_ip = env("API_IP");
        $company_number = env("COMPANY_NUMBER");

        // api user create - salaquzzaman @ 18-10-2024
        $apiname = $request->name;
        $apiemail = $request->email;
        $apipass = $request->password;

        try{
            $response = Http::post($api_ip.'/api/v1/integrate/signUp', [
                'userName' => $apiemail,
                'phone' => '',
                'password' => $apipass,
                'name' => $apiname,
            ]);

            $responseBody = json_decode($response->body(), true);
            if($responseBody && isset($responseBody['message']) && $responseBody['message']=='Success'){

                // success - create user
                $userType = null;
                if ($request->roles) {
                    $userType = $request->roles[0] ?? null;
                }
                $user=User::create($request->all() + ['created_by' => auth()->id(),'type' => $userType]);

                if ($request->roles) {
                    $user->assignRole($request->roles);
                }

                return redirect()->route('users.index')
                    ->with('success', 'User created successfully.');

            }else{
                return redirect()->route('users.create')->with('error', __('User not created.'));
            }

        }catch (\Exception $e) {
            // Log::error($e->getMessage());
            var_dump($e->getMessage());
            exit;
            if($e->getMessage()){
                return redirect()->route('users.create')->with('error', __('User not created.'));
            }
        }


    }

    public function show(User $user)
    {
        if (!auth()->user()->can('View User')) {
            abort(403, 'Sorry !! You are not authorized!');
        }

        return view('user.show', compact('user'));
    }

    public function edit(User $user)
    {
        if (!auth()->user()->can('Edit User')) {
            abort(403, 'Sorry !! You are not authorized!');
        }

        $roles = Role::all();
        return view('user.edit', compact('user','roles'));
    }

    public function update(Request $request, User $user)
    {

        if (!auth()->user()->can('Edit User')) {
            abort(403, 'Sorry !! You are not authorized!');
        }

        $validator = Validator::make($request->all(), [
            'name' => 'required',
            'email' => 'required|email|unique:users,email,' . $user->id,
        ]);

        if ($validator->fails()) {
            return redirect()->route('users.edit', $user)
                ->with('error', 'Missing required fields.')->withInput()->withErrors($validator);
        }

        $userData = $request->except('password');

        if ($request->filled('password')) {
            $userData['password']=$request->password;
        }

        if ($request->roles) {
            $userData['type'] = $request->roles[0] ?? null;
        }

        $user->update($userData);

        $user->roles()->detach();

        if ($request->roles) {
            $user->assignRole($request->roles);
        }

        return redirect()->route('users.index')
            ->with('success', 'User updated successfully.');
    }

    public function destroy(User $user)
    {
        if (!auth()->user()->can('Delete User')) {
            abort(403, 'Sorry !! You are not authorized!');
        }

        $user->delete();

        return redirect()->route('users.index')
            ->with('success', 'User deleted successfully.');
    }

    public function showProfile(){
        $user=User::find(auth()->id());
        return view('profile.index', compact('user'));
    }

    public function updateProfile(Request $request){

        $validator = Validator::make($request->all(), [
            'name' => 'required',
            'email' => 'required|email',
        ]);

        if ($validator->fails()) {
            return redirect()->route('profile')
                ->with('error', 'Missing required fields.');
        }

        $user = $request->all();
        $currentUser = auth()->user();

        if ($request->hasFile('user_image')) {
            if ($currentUser->avatar) {
                Storage::disk('public')->delete('avatar/' . $currentUser->avatar);
            }

            $extension = $request->file('user_image')->getClientOriginalExtension();
            $fileNameToStore = uniqid() . '.' . $extension;
            $path = $request->file('user_image')->storeAs('avatar', $fileNameToStore, 'public');

            $user['avatar'] = $fileNameToStore;

            $avatarPath = 'storage/avatar/' . $fileNameToStore;
            $user_avatar = asset($avatarPath);
            session(['user_avatar' => $user_avatar]);
        }

        $currentUser->update($user);

        return redirect()->route('profile')->with('success', 'Profile updated successfully.');
    }

    public function changePassword(Request $request){


        $token = session('api_token');
        $api_ip = env("API_IP");
        $company_number = env("COMPANY_NUMBER");

        $email = Auth::user()->email;
        $password = trim($request->password);

        if($password==''){
            return redirect()->back()->with('error', 'Sorry, Password is blank.');
        }
        if($token==''){
            return redirect()->back()->with('error', 'API Connection Timeout');
        }

        try{
            $response = Http::withHeaders([
                'Authorization' => 'Bearer ' . $token,
            ])->post($api_ip.'/api/v1/updateUserInfo/erp', [
                'userName' => $email,
                'password' => $password
            ]);

            if ($response->successful()) {
                $responseData = $response->json();
                if($responseData && $responseData['message']=='User updated succesfully'){
                    User::find(auth()->user()->id)->update([
                        'password'=> Hash::make($request->password)
                    ]);
                    return redirect()->route('profile')->with('success', 'Password changed successfully.');
                }else{
                    return redirect()->back()->with('error', 'Sorry, Password Not Reset.');
                }
            }else{
                return redirect()->back()->with('error', 'Sorry, Password Not Reset.');
            }
        }catch (\Exception $e) {
            // Log::error($e->getMessage());
            if($e->getMessage()){
                redirect()->route('dashboard')->with('error', 'API Connection Timeout');
            }
        }

    }

    public function authenticate(Request $request)
    {
        $token = $request->query('token');
        $authToken = DB::table('auth_tokens')->where('token', $token)->first();

        if (!$authToken || Carbon::now()->greaterThan($authToken->expires_at)) {
            return redirect('/login')->with('auth_error', 'Invalid or expired token.');
        }

        $user = User::find($authToken->user_id);
        Auth::login($user);

        if($authToken->acc_token){
            session(['api_token' => $authToken->acc_token]);
        }

        DB::table('auth_tokens')->where('token', $token)->delete();

        return redirect('/');
    }

}

