<?php

namespace App\Http\Controllers;
use App\Models\Customer;
use App\Models\Bill;
use App\Models\BillProcess;
use App\Models\BillRevision;
use App\Models\Collection;
use App\Models\CollectionChd;
use App\Models\CustomerLedger;
use App\Exports\CustomerBillSummaryExport;
use App\Exports\CustomerBillParticularExport;
use App\Exports\BillIncreaseWarningExport;
use Maatwebsite\Excel\Facades\Excel;
use Carbon\Carbon;
use Carbon\CarbonPeriod;
use Illuminate\Support\Facades\DB;
use Barryvdh\DomPDF\Facade\Pdf;
use Illuminate\Http\Request;

class ReportController extends Controller
{
    public function __construct()
    {
        set_time_limit(0);
        ini_set('memory_limit', '2000M');
    }

    // public function customerBillSummary()
    // {
    //     if (!auth()->user()->can('View Report')) {
    //         abort(403, 'Sorry !! You are not authorized!');
    //     }

    //     $customers = Customer::all();

    //     $reportData=[];

    //     foreach ($customers as $customer) {
    //         $paidInvoices = $customer->invoices->filter(function ($invoice) {
    //             return $invoice->payment_status == 'Paid';
    //         });

    //         $unpaidInvoices = $customer->invoices->filter(function ($invoice) {
    //             return $invoice->payment_status == 'Unpaid';
    //         });

    //         $data=[
    //             'customer_id'=>$customer->id,
    //             'customer_name'=>$customer->customer_name,
    //             'total_bill'=>$customer->bills->count(),
    //             'paid_invoice'=>$paidInvoices->count(),
    //             'unpaid_invoice'=>$unpaidInvoices->count(),
    //             'paid_amount'=> $paidInvoices->sum('total_amount'),
    //             'unpaid_amount'=> $unpaidInvoices->sum('total_amount'),
    //         ];
    //         $reportData[]=$data;
    //     }

    //     return view('report.customer-bill-summary', compact('reportData'));
    // }

    public function customerBillSummary()
    {
        if (!auth()->user()->can('View Report')) {
            abort(403, 'Sorry !! You are not authorized!');
        }

        $customers = Customer::all();

        $reportData=[];

        foreach ($customers as $customer) {

            $total_bill = Bill::where('customer_id',$customer->id)->where('approval',1)->where('delete_flag',0)->count();
            $invoices = BillProcess::where('customer_id',$customer->id)->where('delete_flag',0)->whereNotNull('invoice_no')->get();
            $invoice_amount = $invoices->sum('total_amount_vat');
            $paid_amount = 0;
            if($invoices){
                foreach ($invoices as $invoice) {
                    $collectionChd = CollectionChd::where('bill_process_id',$invoice->id)->where('delete_flag',0)->get();
                    $paid_amount = $paid_amount + $collectionChd->sum('collection_amount');
                }
            }


            $data=[
                'customer_id'=>$customer->id,
                'customer_name'=>$customer->customer_name,
                'total_bill'=>$total_bill,
                'total_invoice'=>count($invoices),
                'invoice_amount'=>$invoice_amount,
                'paid_amount'=>$paid_amount,
                'due_amount'=>$invoice_amount-$paid_amount
            ];
            $reportData[]=$data;
        }

        return view('report.customer-bill-summary-new', compact('reportData'));
    }

    public function CustomerBillSummaryExport()
    {
        if (!auth()->user()->can('View Report')) {
            abort(403, 'Sorry !! You are not authorized!');
        }

        $name = 'customer-bill-summary';
        $data = \Excel::download(new CustomerBillSummaryExport(), $name . '.xlsx');

        return $data;
    }

    public function customerWiseBillOKNotGroupBy(Request $request)
    {
        if (!auth()->user()->can('View Report')) {
            abort(403, 'Sorry !! You are not authorized!');
        }

        $bills = [];
        $dateRange = '';
        $grandNetTotal = 0;
        $grandVat = 0;
        $grandTotalAmount = 0;

        $query = BillProcess::query();
        $query->where('delete_flag',0);

        $query->select([
            'id',
            'bill_id',
            'customer_id',
            'invoice_no',
            'invoice_date',
            'bill_subject_desc',
            'total_amount',
            'vat_percent',
            'vat_amount',
            'total_amount_vat',
        ]);

        $query->with([
            'customer:id,customer_name'
        ]);

        if ($request->filled('customer')) {
            $query->where('customer_id', $request->customer);
        }

        if ($request->filled('start_date') && $request->filled('end_date')) {

            $startDate = \DateTime::createFromFormat('d/m/Y', $request->start_date)->format('Y-m-d');
            $query->where(DB::raw("STR_TO_DATE(invoice_date, '%d/%m/%Y')"), '>=', $startDate);

            $endDate = \DateTime::createFromFormat('d/m/Y', $request->end_date)->format('Y-m-d');
            $query->where(DB::raw("STR_TO_DATE(invoice_date, '%d/%m/%Y')"), '<=', $endDate);

            $query->orderBy(DB::raw("STR_TO_DATE(invoice_date, '%d/%m/%Y')"), 'asc');

            $bills = $query->get();
            $grandNetTotal = $bills->sum('total_amount');
            $grandVat = $bills->sum('vat_amount');
            $grandTotalAmount = $bills->sum('total_amount_vat');

            $dateRange = "From ".$request->start_date." To ".$request->end_date;
        }

        $customers = Customer::select('id','customer_name')
            ->orderBy('customer_name', 'asc')
            ->get();

        return view('report.customer-wise-bill', compact('customers', 'bills', 'dateRange','grandNetTotal','grandVat','grandTotalAmount'));

    }

    public function customerWiseBill(Request $request)
    {
        if (!auth()->user()->can('View Report')) {
            abort(403, 'Sorry !! You are not authorized!');
        }

        $bills = [];
        $dateRange = '';
        $grandNetTotal = 0;
        $grandVat = 0;
        $grandTotalAmount = 0;

        $query = BillProcess::query();
        $query->where('delete_flag',0);

        $query->selectRaw("
            customer_id,
            invoice_no,
            invoice_date,
            bill_subject_desc,
            SUM(total_amount) as total_amount,
            vat_percent,
            SUM(vat_amount) as vat_amount,
            SUM(total_amount_vat) as total_amount_vat
        ");

        $query->with([
            'customer:id,customer_name'
        ]);

        if ($request->filled('customer')) {
            $query->where('customer_id', $request->customer);
        }

        if ($request->filled('start_date') && $request->filled('end_date')) {

            $startDate = \DateTime::createFromFormat('d/m/Y', $request->start_date)->format('Y-m-d');
            $endDate = \DateTime::createFromFormat('d/m/Y', $request->end_date)->format('Y-m-d');

            $query->whereRaw("STR_TO_DATE(invoice_date, '%d/%m/%Y') >= ?", [$startDate]);
            $query->whereRaw("STR_TO_DATE(invoice_date, '%d/%m/%Y') <= ?", [$endDate]);

            $query->groupBy([
                'customer_id',
                'invoice_no',
                'invoice_date',
                'bill_subject_desc',
                'vat_percent'
            ]);

            $query->orderByRaw("STR_TO_DATE(invoice_date, '%d/%m/%Y') ASC");

            $bills = $query->get();
            $grandNetTotal = $bills->sum('total_amount');
            $grandVat = $bills->sum('vat_amount');
            $grandTotalAmount = $bills->sum('total_amount_vat');

            $dateRange = "From " . $request->start_date . " To " . $request->end_date;

        }


        $customers = Customer::select('id','customer_name')
            ->orderBy('customer_name', 'asc')
            ->get();

        return view('report.customer-wise-bill', compact('customers', 'bills', 'dateRange','grandNetTotal','grandVat','grandTotalAmount'));

    }


    public function customerWiseBillExportOKnotgroupBy(Request $request, $type)
    {

        $bills = [];
        $dateRange = '';
        $grandNetTotal = 0;
        $grandVat = 0;
        $grandTotalAmount = 0;

        $query = BillProcess::query();
        $query->where('delete_flag',0);

        $query->select([
            'id',
            'customer_id',
            'invoice_no',
            'invoice_date',
            'bill_subject_desc',
            'total_amount',
            'vat_percent',
            'vat_amount',
            'total_amount_vat',
        ]);

        $query->with([
            'customer:id,customer_name'
        ]);

        if ($request->filled('rpt_customer')) {
            $query->where('customer_id', $request->rpt_customer);
        }

        if ($request->filled('rpt_start_date') && $request->filled('rpt_end_date')) {

            $startDate = \DateTime::createFromFormat('d/m/Y', $request->rpt_start_date)->format('Y-m-d');
            $endDate = \DateTime::createFromFormat('d/m/Y', $request->rpt_end_date)->format('Y-m-d');

            $query->where(DB::raw("STR_TO_DATE(invoice_date, '%d/%m/%Y')"), '>=', $startDate);
            $query->where(DB::raw("STR_TO_DATE(invoice_date, '%d/%m/%Y')"), '<=', $endDate);

            $query->orderBy(DB::raw("STR_TO_DATE(invoice_date, '%d/%m/%Y')"), 'asc');

            $bills = $query->get();
            $grandNetTotal = $bills->sum('total_amount');
            $grandVat = $bills->sum('vat_amount');
            $grandTotalAmount = $bills->sum('total_amount_vat');

            $dateRange = "From " . $request->rpt_start_date . " To " . $request->rpt_end_date;
        }

        $data['bills'] = $bills;
        $data['dateRange'] = $dateRange;
        $data['customer'] = $request->rpt_customer;
        $data['grandNetTotal'] = $grandNetTotal;
        $data['grandVat'] = $grandVat;
        $data['grandTotalAmount'] = $grandTotalAmount;

        if($type=='pdf'){
            $name = "Customer_Wise_Bill_".date('Y-m-d').".pdf";
            $pdf = Pdf::loadView('report.pdf.customer-wise-bill-pdf', $data);
            $pdf->setPaper('A4', 'landscape');
            // return $pdf->stream($name, ['Attachment'=>false]);
            return $pdf->download($name); // Force download
        }

    }

    public function customerWiseBillExport(Request $request, $type)
    {

        $bills = [];
        $dateRange = '';
        $grandNetTotal = 0;
        $grandVat = 0;
        $grandTotalAmount = 0;

        $query = BillProcess::query();
        $query->where('delete_flag', 0);

        $query->selectRaw("
            customer_id,
            invoice_no,
            invoice_date,
            bill_subject_desc,
            vat_percent,
            SUM(total_amount) as total_amount,
            SUM(vat_amount) as vat_amount,
            SUM(total_amount_vat) as total_amount_vat
        ");

        $query->with(['customer:id,customer_name']);

        if ($request->filled('rpt_customer')) {
            $query->where('customer_id', $request->rpt_customer);
        }

        if ($request->filled('rpt_start_date') && $request->filled('rpt_end_date')) {

            $startDate = \DateTime::createFromFormat('d/m/Y', $request->rpt_start_date)->format('Y-m-d');
            $endDate = \DateTime::createFromFormat('d/m/Y', $request->rpt_end_date)->format('Y-m-d');

            $query->whereRaw("STR_TO_DATE(invoice_date, '%d/%m/%Y') >= ?", [$startDate]);
            $query->whereRaw("STR_TO_DATE(invoice_date, '%d/%m/%Y') <= ?", [$endDate]);

            $query->groupBy([
                'customer_id',
                'invoice_no',
                'invoice_date',
                'bill_subject_desc',
                'vat_percent'
            ]);

            $query->orderByRaw("STR_TO_DATE(invoice_date, '%d/%m/%Y') ASC");

            $bills = $query->get();
            $grandNetTotal = $bills->sum('total_amount');
            $grandVat = $bills->sum('vat_amount');
            $grandTotalAmount = $bills->sum('total_amount_vat');

            $dateRange = "From " . $request->rpt_start_date . " To " . $request->rpt_end_date;
        }

        $data['bills'] = $bills;
        $data['dateRange'] = $dateRange;
        $data['customer'] = $request->rpt_customer;
        $data['grandNetTotal'] = $grandNetTotal;
        $data['grandVat'] = $grandVat;
        $data['grandTotalAmount'] = $grandTotalAmount;

        // Generate PDF
        if ($type == 'pdf') {
            $name = "Customer_Wise_Bill_" . date('Y-m-d') . ".pdf";
            $pdf = Pdf::loadView('report.pdf.customer-wise-bill-pdf', $data);
            $pdf->setPaper('A4', 'landscape');
            // return $pdf->stream($name, ['Attachment'=>false]);
            return $pdf->download($name);
        }

    }

    public function customerWiseBillExport333(Request $request)
    {
        if (!auth()->user()->can('View Report')) {
            abort(403, 'Sorry !! You are not authorized!');
        }

        return Excel::download(new CustomerWiseBillExport($request), 'customer-wise-bill.xlsx');
    }

    public function customerCollectionBillSummary(Request $request)
    {
        if (!auth()->user()->can('View Report')) {
            abort(403, 'Sorry !! You are not authorized!');
        }

        $collections = [];
        $dateRange = '';

        $totalReceived = 0;
        $totalTDS = 0;
        $totalVAT = 0;

        $query = Collection::query();
        $query->where('delete_flag', 0);

        $query->select([
            'id',
            'customer_id',
            'ref_no',
            'coll_date',
            'instrument_mode',
            'instrument_no',
            'instrument_date',
            'bank_id',
            'money_receipt_no',
            'money_receipt_date',
            'bank_slip_no',
            'bank_deposit_date',
            'received_amount',
            'tds_amount',
            'vat_amount',
        ]);

        $query->with('customer:id,customer_name');
        $query->with('customerBank:id,bank_name');

        if ($request->filled('customer')) {
            $query->where('customer_id', $request->customer);
        }

        if ($request->filled('start_date') && $request->filled('end_date')) {

            $startDate = \DateTime::createFromFormat('d/m/Y', $request->start_date)->format('Y-m-d');
            $endDate = \DateTime::createFromFormat('d/m/Y', $request->end_date)->format('Y-m-d');

            $query->where(DB::raw("STR_TO_DATE(coll_date, '%d/%m/%Y')"), '>=', $startDate);
            $query->where(DB::raw("STR_TO_DATE(coll_date, '%d/%m/%Y')"), '<=', $endDate);

            $query->orderBy('ref_no');
            $collections = $query->get();

            if ($collections->isNotEmpty()) {
                $totalReceived = $collections->sum('received_amount');
                $totalTDS = $collections->sum('tds_amount');
                $totalVAT = $collections->sum('vat_amount');
            }

            $dateRange = "From " . $request->start_date . " To " . $request->end_date;
        }

        $customers = Customer::select('id','customer_name')
            ->orderBy('customer_name', 'asc')
            ->get();




            // $query = Collection::query();
            // $query->where('delete_flag', 0);

            // $query->select([
            //     'collections.id',
            //     'collections.customer_id',
            //     'collections.ref_no',
            //     'collections.coll_date',
            //     'collections.instrument_mode',
            //     'collections.instrument_no',
            //     'collections.instrument_date',
            //     'collections.dep_bank_id',
            //     'collections.money_receipt_no',
            //     'collections.money_receipt_date',
            //     'collections.bank_slip_no',
            //     'collections.bank_deposit_date',
            //     'collections.received_amount',
            //     'collections.tds_amount',
            //     'collections.vat_amount',
            //     'billing_customers.customer_name', // Add customer_name to the select
            // ]);

            // $query->join('billing_customers', 'collections.customer_id', '=', 'billing_customers.id'); // Join the customers table

            // if ($request->filled('customer')) {
            //     $query->where('collections.customer_id', $request->customer);
            // }

            // if ($request->filled('start_date') && $request->filled('end_date')) {
            //     $startDate = Carbon::createFromFormat('d/m/Y', $this->normalizeDate($request->start_date))->format('Y-m-d');
            //     $endDate = Carbon::createFromFormat('d/m/Y', $this->normalizeDate($request->end_date))->format('Y-m-d');

            //     $query->where(DB::raw("STR_TO_DATE(collections.bank_deposit_date, '%d/%m/%Y')"), '>=', $startDate);
            //     $query->where(DB::raw("STR_TO_DATE(collections.bank_deposit_date, '%d/%m/%Y')"), '<=', $endDate);
            // }

            // // Add ordering
            // $query->orderBy('billing_customers.customer_name', 'asc') // Order by customer_name
            //     ->orderBy(DB::raw("STR_TO_DATE(collections.bank_deposit_date, '%d/%m/%Y')"), 'asc'); // Order by bank_deposit_date

            // $collections = $query->get();

            // if ($collections->isNotEmpty()) {
            //     $totalReceived = $collections->sum('received_amount');
            //     $totalTDS = $collections->sum('tds_amount');
            //     $totalVAT = $collections->sum('vat_amount');
            // }

            // $dateRange = "From " . $request->start_date . " To " . $request->end_date;

        return view('report.customer-collection-bill', compact('collections','customers','dateRange','totalReceived','totalTDS','totalVAT'));
    }

    public function customerCollectionSummaryExport(Request $request, $type)
    {
        // set_time_limit(0);
        // ini_set('max_execution_time', 300);
        // ini_set('memory_limit', '512M');

        $collections = [];
        $dateRange = '';

        $totalReceived = 0;
        $totalTDS = 0;
        $totalVAT = 0;

        $query = Collection::query();
        $query->where('delete_flag', 0);

        $query->select([
            'id',
            'customer_id',
            'ref_no',
            'coll_date',
            'instrument_mode',
            'instrument_no',
            'instrument_date',
            'bank_id',
            'money_receipt_no',
            'money_receipt_date',
            'bank_slip_no',
            'bank_deposit_date',
            'received_amount',
            'tds_amount',
            'vat_amount',
        ]);

        $query->with('customer:id,customer_name');
        $query->with('customerBank:id,bank_name');

        if ($request->filled('rpt_customer')) {
            $query->where('customer_id', $request->rpt_customer);
        }

        if ($request->filled('rpt_start_date') && $request->filled('rpt_end_date')) {

            $startDate = \DateTime::createFromFormat('d/m/Y', $request->rpt_start_date)->format('Y-m-d');
            $endDate = \DateTime::createFromFormat('d/m/Y', $request->rpt_end_date)->format('Y-m-d');

            $query->where(DB::raw("STR_TO_DATE(coll_date, '%d/%m/%Y')"), '>=', $startDate);
            $query->where(DB::raw("STR_TO_DATE(coll_date, '%d/%m/%Y')"), '<=', $endDate);

            $query->orderBy('ref_no');
            $collections = $query->get();

            if ($collections->isNotEmpty()) {
                $totalReceived = $collections->sum('received_amount');
                $totalTDS = $collections->sum('tds_amount');
                $totalVAT = $collections->sum('vat_amount');
            }

            $dateRange = "From " . $request->rpt_start_date . " To " . $request->rpt_end_date;
        }

        $data['collections'] = $collections;
        $data['dateRange'] = $dateRange;
        $data['customer'] = $request->rpt_customer;
        $data['totalReceived'] = $totalReceived;
        $data['totalTDS'] = $totalTDS;
        $data['totalVAT'] = $totalVAT;

        if($type=='pdf'){
            $name = "Collection_Register_".date('Y-m-d').".pdf";
            $pdf = Pdf::loadView('report.pdf.customer-collection-summary-pdf', $data);
            $pdf->setPaper('A4', 'landscape');
            // return $pdf->stream($name, ['Attachment'=>false]);
            return $pdf->download($name); // Force download
        }

    }

    public function customerCollectionBillExport(Request $request)
    {
        if (!auth()->user()->can('View Report')) {
            abort(403, 'Sorry !! You are not authorized!');
        }

        return Excel::download(new CustomerCollectionBillExport($request), 'customer-collection-bill.xlsx');
    }


    public function customerBillParticular(Request $request)
    {
        if (!auth()->user()->can('View Report')) {
            abort(403, 'Sorry !! You are not authorized!');
        }

        $bills = [];
        $query = Bill::query();

        $query->select([
            'billing_bills.id as bill_no',
            'customer_id',
            'billDivName',
            'bill_subject_id',
            'particulars',
            'ref_no',
            'ref_date',
            'qty',
            'unit_price_vat',
            'vat_status',
            'vat_percent',
            'installation_date',
            'bill_type',
            'start_date',
            'yearly_increase_percent',
            'increase_start_date',
            'end_date',
            'bill_group',
            'billing_customers.status',
            'billing_customers.customer_name',
        ]);

        $query->join('billing_customers', 'billing_bills.customer_id', '=', 'billing_customers.id');
        $query->where('billing_customers.status', 'Active');
        $query->where('billing_bills.approval', 1);
        $query->where('billing_bills.delete_flag', 0);

        if ($request->filled('customer')) {
            $query->where('customer_id', $request->customer);
            $query->orderBy('billing_bills.id', 'asc');
        }else{
            $query->orderBy('billing_customers.customer_name', 'asc')
                ->orderBy('billing_bills.id', 'asc');
        }

        if ($request->filled('status')) {
            $query->where('billing_bills.status', $request->status);
            $query->with(['billSubject:id,bill_subject']);
            $bills = $query->get();
        }

        $customers = Customer::select('id','customer_name')
            ->orderBy('customer_name', 'asc')
            ->get();

        return view('report.customer-bill-particular', compact('bills','customers'));
    }

    public function customerBillParticularExport(Request $request, $type)
    {

        $bills = [];
        $query = Bill::query();

        $query->select([
            'billing_bills.id as bill_no',
            'customer_id',
            'billDivName',
            'bill_subject_id',
            'particulars',
            'ref_no',
            'ref_date',
            'qty',
            'unit_price_vat',
            'vat_status',
            'vat_percent',
            'installation_date',
            'bill_type',
            'start_date',
            'yearly_increase_percent',
            'increase_start_date',
            'end_date',
            'bill_group',
            'billing_customers.status',
            'billing_customers.customer_name',
        ]);

        $query->join('billing_customers', 'billing_bills.customer_id', '=', 'billing_customers.id');
        $query->where('billing_customers.status', 'Active');
        $query->where('billing_bills.approval', 1);
        $query->where('billing_bills.delete_flag', 0);

        if ($request->filled('rpt_customer')) {
            $query->where('customer_id', $request->rpt_customer);
            $query->orderBy('billing_bills.id', 'asc');
        }else{
            $query->orderBy('billing_customers.customer_name', 'asc')
                ->orderBy('billing_bills.id', 'asc');
        }

        if ($request->filled('rpt_status')) {
            $query->where('billing_bills.status', $request->rpt_status);
            $query->with(['billSubject:id,bill_subject']);
            $bills = $query->get();
        }

        $data['bills'] = $bills;
        $data['customer'] = $request->rpt_customer;
        $data['status'] = $request->rpt_status;

        if($type=='pdf'){
            $name = "Billing_Particulars_for_Customer_".date('Y-m-d').".pdf";
            $pdf = Pdf::loadView('report.pdf.customer-bill-particular-pdf', $data);
            $pdf->setPaper('Legal', 'landscape');
            // return $pdf->stream($name, ['Attachment'=>false]);
            return $pdf->download($name); // Force download
        }

        if($type=='excel'){
            $name = "Billing_Particulars_for_Customer_".date('Y-m-d').".xlsx";
            return Excel::download(new CustomerBillParticularExport($request), $name);
        }

    }

    public function customerDueReport(Request $request)
    {
        $bills = collect();
        $customerInfo = null;
        $currency = '';
        $totalDue = 0;

        $query = BillProcess::query();
        $query->select(
            'invoice_no',
            'invoice_date',
            'currency',
            DB::raw('SUM(due) as total_due')
        );
        $query->where('delete_flag', 0)
            ->where('payment_status', '!=', 'Paid');

        if ($request->filled('customer')) {
            $customerInfo = Customer::select('id','customer_name', 'address', 'phone', 'bin_no')->find($request->customer);

            if ($customerInfo) {
                $query->where('customer_id', $customerInfo->id);

                if ($request->filled('start_date')) {
                    $startDate = Carbon::createFromFormat('d/m/Y', $this->normalizeDate($request->start_date))->format('Y-m-d');
                    $query->where(DB::raw("STR_TO_DATE(invoice_date, '%d/%m/%Y')"), '>=', $startDate);
                }

                if ($request->filled('end_date')) {
                    $endDate = Carbon::createFromFormat('d/m/Y', $this->normalizeDate($request->end_date))->format('Y-m-d');
                    $query->where(DB::raw("STR_TO_DATE(invoice_date, '%d/%m/%Y')"), '<=', $endDate);
                }

                $query->groupBy('invoice_no', 'invoice_date', 'currency')
                    ->orderByRaw("STR_TO_DATE(invoice_date, '%d/%m/%Y') ASC");
                $bills = $query->get();

                if ($bills->isNotEmpty()) {
                    $currency = $bills->first()->currency;
                    $totalDue = $bills->sum('total_due');
                }
            }
        }

        $customers = Customer::select('id','customer_name')
            ->orderBy('customer_name', 'asc')
            ->get();

        return view('report.customer-due-report', compact('bills','customers','customerInfo','currency','totalDue'));

    }

    private function normalizeDate($date)
    {
        $formats = [
            'd/m/y',
            'd/m/Y',
            'd-M-Y',
            'd-M-y'
        ];

        foreach ($formats as $format) {
            try {
                $parsedDate = \Carbon\Carbon::createFromFormat($format, $date);
                return $parsedDate->format('d/m/Y');
            } catch (\Exception $e) {
            }
        }

        return $date;
    }





 //old correct code for ageing report
    // public function customerOutstandingAgeingBill(Request $request)
    // {
    //     if ($request->has('start_date')) {

    //         $startDate = Carbon::createFromFormat('m/Y', $request->input('start_date'))->startOfMonth();

    //         $pastMonths = [];

    //         for ($i = 0; $i < 12; $i++) {
    //             $pastMonths[] = [
    //                 'month' => $startDate->format('F'),
    //                 'year' => $startDate->format('Y'),
    //                 'month_year' => $startDate->format('F Y')
    //             ];
    //             $startDate->subMonth();
    //         }
    //         $pastMonths = array_reverse($pastMonths);

    //         $customers = Customer::select('id', 'customer_name')->get();

    //         $monthsYears = array_map(function ($monthYear) {
    //             return $monthYear['month'] . ' ' . $monthYear['year'];
    //         }, $pastMonths);

    //         $bills = BillProcess::whereIn(
    //                 DB::raw("CONCAT(month, ' ', year)"),
    //                 $monthsYears
    //             )
    //             ->select('customer_id', 'month', 'year', DB::raw('SUM(total_amount_vat) as total_amount_vat'))
    //             ->groupBy('customer_id', 'month', 'year')
    //             ->get();

    //         $billsByCustomer = [];
    //         foreach ($bills as $bill) {
    //             $monthYearKey = $bill->month . ' ' . $bill->year;
    //             $billsByCustomer[$bill->customer_id][$monthYearKey] = $bill->total_amount_vat;
    //         }

    //         $reportData = [];
    //         foreach ($customers as $customer) {
    //             $data = [
    //                 'id' => $customer->id,
    //                 'name' => $customer->customer_name,
    //                 'bills' => []
    //             ];

    //             foreach ($pastMonths as $date) {
    //                 $monthYearKey = $date['month'] . ' ' . $date['year'];
    //                 $total_bill = $billsByCustomer[$customer->id][$monthYearKey] ?? 0;

    //                 $data['bills'][] = [
    //                     'total_bill' => $total_bill,
    //                     'month' => $date['month'],
    //                     'year' => $date['year']
    //                 ];
    //             }

    //             $reportData[] = $data;
    //         }

    //         // return $reportData;

    //         return view('report.customer-outstanding-ageing-bill', compact('reportData', 'pastMonths'));

    //     }

    //     return view('report.customer-outstanding-ageing-bill');
    // }





   //multiple customer select &  month year wise serach done (1)

    // public function customerOutstandingAgeingBill(Request $request)
    // {
    //     $allCustomers = Customer::select('id', 'customer_name')->get();

    //     if ($request->has('start_date')) {

    //         $startDate = Carbon::createFromFormat('m/Y', $request->input('start_date'))->startOfMonth();

    //         $pastMonths = [];

    //         for ($i = 0; $i < 12; $i++) {
    //             $pastMonths[] = [
    //                 'month' => $startDate->format('F'),
    //                 'year' => $startDate->format('Y'),
    //                 'month_year' => $startDate->format('F Y')
    //             ];
    //             $startDate->subMonth();
    //         }
    //         $pastMonths = array_reverse($pastMonths);

    //         $monthsYears = array_map(function ($monthYear) {
    //             return $monthYear['month'] . ' ' . $monthYear['year'];
    //         }, $pastMonths);

    //         $billsQuery = BillProcess::where('delete_flag', 0)
    //             ->whereIn(DB::raw("CONCAT(month, ' ', year)"), $monthsYears)
    //             ->select('customer_id', 'month', 'year', DB::raw('SUM(total_amount_vat) as total_amount_vat'))
    //             ->groupBy('customer_id', 'month', 'year');

    //         if ($request->filled('customer_id')) {
    //             $selectedCustomerIds = $request->input('customer_id');
    //             $billsQuery->whereIn('customer_id', $selectedCustomerIds);

    //             $customers = Customer::whereIn('id', $selectedCustomerIds)->select('id', 'customer_name')->get();
    //         } else {
    //             $customers = Customer::select('id', 'customer_name')->get();
    //         }

    //         $bills = $billsQuery->get();

    //         $billsByCustomer = [];
    //         foreach ($bills as $bill) {
    //             $monthYearKey = $bill->month . ' ' . $bill->year;
    //             $billsByCustomer[$bill->customer_id][$monthYearKey] = $bill->total_amount_vat;
    //         }

    //         $reportData = [];
    //         foreach ($customers as $customer) {
    //             $data = [
    //                 'id' => $customer->id,
    //                 'name' => $customer->customer_name,
    //                 'bills' => []
    //             ];

    //             foreach ($pastMonths as $date) {
    //                 $monthYearKey = $date['month'] . ' ' . $date['year'];
    //                 $total_bill = $billsByCustomer[$customer->id][$monthYearKey] ?? 0;

    //                 $data['bills'][] = [
    //                     'total_bill' => $total_bill,
    //                     'month' => $date['month'],
    //                     'year' => $date['year']
    //                 ];
    //             }

    //             $reportData[] = $data;
    //         }

    //         return view('report.customer-outstanding-ageing-bill', compact('reportData', 'pastMonths', 'allCustomers'));
    //     }

    //     return view('report.customer-outstanding-ageing-bill', compact('allCustomers'));
    // }


    public function customerOutstandingAgeingBill(Request $request)
    {

        $allCustomers = Customer::select('id','customer_name')
            ->orderBy('customer_name', 'asc')
            ->get();

        if ($request->has('start_date')) {

            $startDate = Carbon::createFromFormat('d/m/Y', $request->input('start_date'))->format('Y-m-d');

            $condition = '';
            if ($request->filled('customer_id')) {
                $selectedCustomerIds = $request->input('customer_id');

                if (in_array(80000000, $selectedCustomerIds)) {
                } else {
                    $cust = implode(',',$selectedCustomerIds);
                    $condition = " AND customer_id in ($cust) ";
                }
            }

            $reportData = DB::select("SELECT
                    customer_id AS customerID,
                    currency,
                    SUM(CASE WHEN DATE_FORMAT(created_at, '%Y-%m') = DATE_FORMAT(LAST_DAY('$startDate'), '%Y-%m') THEN due ELSE 0 END) AS `00 Month`,
                    SUM(CASE WHEN DATE_FORMAT(created_at, '%Y-%m') = DATE_FORMAT(DATE_SUB(LAST_DAY('$startDate'), INTERVAL 1 MONTH), '%Y-%m') THEN due ELSE 0 END) AS `01 Month`,
                    SUM(CASE WHEN DATE_FORMAT(created_at, '%Y-%m') = DATE_FORMAT(DATE_SUB(LAST_DAY('$startDate'), INTERVAL 2 MONTH), '%Y-%m') THEN due ELSE 0 END) AS `02 Month`,
                    SUM(CASE WHEN DATE_FORMAT(created_at, '%Y-%m') = DATE_FORMAT(DATE_SUB(LAST_DAY('$startDate'), INTERVAL 3 MONTH), '%Y-%m') THEN due ELSE 0 END) AS `03 Month`,
                    SUM(CASE WHEN DATE_FORMAT(created_at, '%Y-%m') = DATE_FORMAT(DATE_SUB(LAST_DAY('$startDate'), INTERVAL 4 MONTH), '%Y-%m') THEN due ELSE 0 END) AS `04 Month`,
                    SUM(CASE WHEN DATE_FORMAT(created_at, '%Y-%m') = DATE_FORMAT(DATE_SUB(LAST_DAY('$startDate'), INTERVAL 5 MONTH), '%Y-%m') THEN due ELSE 0 END) AS `05 Month`,
                    SUM(CASE WHEN DATE_FORMAT(created_at, '%Y-%m') = DATE_FORMAT(DATE_SUB(LAST_DAY('$startDate'), INTERVAL 6 MONTH), '%Y-%m') THEN due ELSE 0 END) AS `06 Month`,
                    SUM(CASE WHEN DATE_FORMAT(created_at, '%Y-%m') = DATE_FORMAT(DATE_SUB(LAST_DAY('$startDate'), INTERVAL 7 MONTH), '%Y-%m') THEN due ELSE 0 END) AS `07 Month`,
                    SUM(CASE WHEN DATE_FORMAT(created_at, '%Y-%m') = DATE_FORMAT(DATE_SUB(LAST_DAY('$startDate'), INTERVAL 8 MONTH), '%Y-%m') THEN due ELSE 0 END) AS `08 Month`,
                    SUM(CASE WHEN DATE_FORMAT(created_at, '%Y-%m') = DATE_FORMAT(DATE_SUB(LAST_DAY('$startDate'), INTERVAL 9 MONTH), '%Y-%m') THEN due ELSE 0 END) AS `09 Month`,
                    SUM(CASE WHEN DATE_FORMAT(created_at, '%Y-%m') = DATE_FORMAT(DATE_SUB(LAST_DAY('$startDate'), INTERVAL 10 MONTH), '%Y-%m') THEN due ELSE 0 END) AS `10 Month`,
                    SUM(CASE WHEN DATE_FORMAT(created_at, '%Y-%m') = DATE_FORMAT(DATE_SUB(LAST_DAY('$startDate'), INTERVAL 11 MONTH), '%Y-%m') THEN due ELSE 0 END) AS `11 Month`,
                    SUM(CASE WHEN DATE_FORMAT(created_at, '%Y-%m') = DATE_FORMAT(DATE_SUB(LAST_DAY('$startDate'), INTERVAL 12 MONTH), '%Y-%m') THEN due ELSE 0 END) AS `12 Month`,
                    SUM(CASE WHEN created_at < DATE_SUB(LAST_DAY('$startDate'), INTERVAL 13 MONTH) THEN due ELSE 0 END) AS `>12 Months`
                FROM
                    bill_processes
                WHERE
                    created_at <= LAST_DAY('$startDate')
                    AND delete_flag = 0
                    $condition
                GROUP BY
                    customer_id,currency
                ORDER BY
                    customer_id
            ");

            return view('report.customer-outstanding-ageing-bill', compact('reportData', 'allCustomers'));
        }

        return view('report.customer-outstanding-ageing-bill', compact('allCustomers'));
    }


    // OK WORKING BUT 3 month data overlap to next column
    // public function customerOutstandingAgeingBillContact(Request $request)
    // {

    //     $allCustomers = Customer::select('id','customer_name')
    //         ->orderBy('customer_name', 'asc')
    //         ->get();

    //     if ($request->has('start_date')) {

    //         $startDate = Carbon::createFromFormat('d/m/Y', $request->input('start_date'))->format('Y-m-d');

    //         $condition = '';
    //         if ($request->filled('customer_id')) {
    //             $selectedCustomerIds = $request->input('customer_id');

    //             if (in_array(80000000, $selectedCustomerIds)) {
    //             } else {
    //                 $cust = implode(',',$selectedCustomerIds);
    //                 $condition = " AND customer_id in ($cust) ";
    //             }
    //         }

    //         $reportData = DB::select("SELECT
    //                 customer_id AS customerID,
    //                 currency,
    //                 SUM(CASE WHEN DATE_FORMAT(created_at, '%Y-%m') = DATE_FORMAT(LAST_DAY('$startDate'), '%Y-%m') THEN due ELSE 0 END) AS `00 Month`,
    //                 SUM(CASE WHEN DATE_FORMAT(created_at, '%Y-%m') = DATE_FORMAT(DATE_SUB(LAST_DAY('$startDate'), INTERVAL 1 MONTH), '%Y-%m') THEN due ELSE 0 END) AS `01 Month`,
    //                 SUM(CASE WHEN DATE_FORMAT(created_at, '%Y-%m') = DATE_FORMAT(DATE_SUB(LAST_DAY('$startDate'), INTERVAL 2 MONTH), '%Y-%m') THEN due ELSE 0 END) AS `02 Month`,
    //                 SUM(CASE WHEN DATE_FORMAT(created_at, '%Y-%m') = DATE_FORMAT(DATE_SUB(LAST_DAY('$startDate'), INTERVAL 3 MONTH), '%Y-%m') THEN due ELSE 0 END) AS `03 Month`,
    //                 -- Sum for records between >3 months and <13 months
    //                 SUM(CASE WHEN created_at < DATE_SUB(LAST_DAY('$startDate'), INTERVAL 3 MONTH)
    //                         AND created_at >= DATE_SUB(LAST_DAY('$startDate'), INTERVAL 13 MONTH)
    //                         THEN due ELSE 0 END) AS `>3 Month & <13`,
    //                 SUM(CASE WHEN created_at < DATE_SUB(LAST_DAY('$startDate'), INTERVAL 13 MONTH) THEN due ELSE 0 END) AS `>12 Months`
    //             FROM
    //                 bill_processes
    //             WHERE
    //                 created_at <= LAST_DAY('$startDate')
    //                 AND delete_flag = 0
    //                 $condition
    //             GROUP BY
    //                 customer_id,currency
    //             ORDER BY
    //                 customer_id
    //         ");

    //         return view('report.customer-outstanding-ageing-bill-contact', compact('reportData', 'allCustomers'));
    //     }

    //     return view('report.customer-outstanding-ageing-bill-contact', compact('allCustomers'));
    // }


    public function customerOutstandingAgeingBillContact(Request $request)
    {

        $allCustomers = Customer::select('id','customer_name')
            ->orderBy('customer_name', 'asc')
            ->get();

        if ($request->has('start_date')) {

            $startDate = Carbon::createFromFormat('d/m/Y', $request->input('start_date'))->format('Y-m-d');

            $condition = '';
            if ($request->filled('customer_id')) {
                $selectedCustomerIds = $request->input('customer_id');

                if (in_array(80000000, $selectedCustomerIds)) {
                } else {
                    $cust = implode(',',$selectedCustomerIds);
                    $condition = " AND customer_id in ($cust) ";
                }
            }


            $reportData = DB::select("SELECT
                    customer_id AS customerID,
                    currency,

                    -- Exact month buckets (00–12), using your proven pattern
                    SUM(CASE WHEN DATE_FORMAT(created_at, '%Y-%m') = DATE_FORMAT(LAST_DAY('$startDate'), '%Y-%m') THEN due ELSE 0 END) AS `00 Month`,
                    SUM(CASE WHEN DATE_FORMAT(created_at, '%Y-%m') = DATE_FORMAT(DATE_SUB(LAST_DAY('$startDate'), INTERVAL 1 MONTH), '%Y-%m') THEN due ELSE 0 END) AS `01 Month`,
                    SUM(CASE WHEN DATE_FORMAT(created_at, '%Y-%m') = DATE_FORMAT(DATE_SUB(LAST_DAY('$startDate'), INTERVAL 2 MONTH), '%Y-%m') THEN due ELSE 0 END) AS `02 Month`,
                    SUM(CASE WHEN DATE_FORMAT(created_at, '%Y-%m') = DATE_FORMAT(DATE_SUB(LAST_DAY('$startDate'), INTERVAL 3 MONTH), '%Y-%m') THEN due ELSE 0 END) AS `03 Month`,
                    SUM(CASE WHEN DATE_FORMAT(created_at, '%Y-%m') = DATE_FORMAT(DATE_SUB(LAST_DAY('$startDate'), INTERVAL 4 MONTH), '%Y-%m') THEN due ELSE 0 END) AS `04 Month`,
                    SUM(CASE WHEN DATE_FORMAT(created_at, '%Y-%m') = DATE_FORMAT(DATE_SUB(LAST_DAY('$startDate'), INTERVAL 5 MONTH), '%Y-%m') THEN due ELSE 0 END) AS `05 Month`,
                    SUM(CASE WHEN DATE_FORMAT(created_at, '%Y-%m') = DATE_FORMAT(DATE_SUB(LAST_DAY('$startDate'), INTERVAL 6 MONTH), '%Y-%m') THEN due ELSE 0 END) AS `06 Month`,
                    SUM(CASE WHEN DATE_FORMAT(created_at, '%Y-%m') = DATE_FORMAT(DATE_SUB(LAST_DAY('$startDate'), INTERVAL 7 MONTH), '%Y-%m') THEN due ELSE 0 END) AS `07 Month`,
                    SUM(CASE WHEN DATE_FORMAT(created_at, '%Y-%m') = DATE_FORMAT(DATE_SUB(LAST_DAY('$startDate'), INTERVAL 8 MONTH), '%Y-%m') THEN due ELSE 0 END) AS `08 Month`,
                    SUM(CASE WHEN DATE_FORMAT(created_at, '%Y-%m') = DATE_FORMAT(DATE_SUB(LAST_DAY('$startDate'), INTERVAL 9 MONTH), '%Y-%m') THEN due ELSE 0 END) AS `09 Month`,
                    SUM(CASE WHEN DATE_FORMAT(created_at, '%Y-%m') = DATE_FORMAT(DATE_SUB(LAST_DAY('$startDate'), INTERVAL 10 MONTH), '%Y-%m') THEN due ELSE 0 END) AS `10 Month`,
                    SUM(CASE WHEN DATE_FORMAT(created_at, '%Y-%m') = DATE_FORMAT(DATE_SUB(LAST_DAY('$startDate'), INTERVAL 11 MONTH), '%Y-%m') THEN due ELSE 0 END) AS `11 Month`,
                    SUM(CASE WHEN DATE_FORMAT(created_at, '%Y-%m') = DATE_FORMAT(DATE_SUB(LAST_DAY('$startDate'), INTERVAL 12 MONTH), '%Y-%m') THEN due ELSE 0 END) AS `12 Month`,

                    -- 4–12 months bucket (sum of months 04..12; no overlap with 03 Month)
                    SUM(
                        CASE
                            WHEN DATE_FORMAT(created_at, '%Y-%m') BETWEEN
                                DATE_FORMAT(DATE_SUB(LAST_DAY('$startDate'), INTERVAL 12 MONTH), '%Y-%m')
                                AND DATE_FORMAT(DATE_SUB(LAST_DAY('$startDate'), INTERVAL 4 MONTH), '%Y-%m')
                            THEN due ELSE 0
                        END
                    ) AS `>3 Month & <13`,

                    -- Older than 12 months (13+ months)
                    SUM(CASE WHEN created_at < DATE_SUB(LAST_DAY('$startDate'), INTERVAL 13 MONTH) THEN due ELSE 0 END) AS `>12 Months`

                FROM bill_processes
                WHERE
                    created_at <= LAST_DAY('$startDate')
                    AND delete_flag = 0
                    $condition
                GROUP BY customer_id, currency
                ORDER BY customer_id
            ");


            return view('report.customer-outstanding-ageing-bill-contact', compact('reportData', 'allCustomers'));
        }

        return view('report.customer-outstanding-ageing-bill-contact', compact('allCustomers'));
    }

    public function customerOutstandingAgeingBillDept(Request $request)
    {

        $allCustomers = Customer::select('id','customer_name')
            ->orderBy('customer_name', 'asc')
            ->get();

        if ($request->has('start_date')) {

            $startDate = Carbon::createFromFormat('d/m/Y', $request->input('start_date'))->format('Y-m-d');

            $condition = '';
            if ($request->filled('customer_id')) {
                $selectedCustomerIds = $request->input('customer_id');

                if (in_array(80000000, $selectedCustomerIds)) {
                } else {
                    $cust = implode(',',$selectedCustomerIds);
                    $condition = " AND customer_id in ($cust) ";
                }
            }

            // $reportData444 = DB::select("SELECT
            //         customer_id AS customerID,
            //         department_id,
            //         billDivName,
            //         currency,
            //         SUM(CASE WHEN DATE_FORMAT(created_at, '%Y-%m') = DATE_FORMAT(LAST_DAY('$startDate'), '%Y-%m') THEN due ELSE 0 END) AS `00 Month`,
            //         SUM(CASE WHEN DATE_FORMAT(created_at, '%Y-%m') = DATE_FORMAT(DATE_SUB(LAST_DAY('$startDate'), INTERVAL 1 MONTH), '%Y-%m') THEN due ELSE 0 END) AS `01 Month`,
            //         SUM(CASE WHEN DATE_FORMAT(created_at, '%Y-%m') = DATE_FORMAT(DATE_SUB(LAST_DAY('$startDate'), INTERVAL 2 MONTH), '%Y-%m') THEN due ELSE 0 END) AS `02 Month`,
            //         SUM(CASE WHEN DATE_FORMAT(created_at, '%Y-%m') = DATE_FORMAT(DATE_SUB(LAST_DAY('$startDate'), INTERVAL 3 MONTH), '%Y-%m') THEN due ELSE 0 END) AS `03 Month`,
            //         -- Sum for records between >3 months and <13 months
            //         SUM(CASE WHEN created_at < DATE_SUB(LAST_DAY('$startDate'), INTERVAL 3 MONTH)
            //                 AND created_at >= DATE_SUB(LAST_DAY('$startDate'), INTERVAL 13 MONTH)
            //                 THEN due ELSE 0 END) AS `>3 Month & <13`,
            //         SUM(CASE WHEN created_at < DATE_SUB(LAST_DAY('$startDate'), INTERVAL 13 MONTH) THEN due ELSE 0 END) AS `>12 Months`
            //     FROM
            //         bill_processes
            //     WHERE
            //         created_at <= LAST_DAY('$startDate')
            //         AND delete_flag = 0
            //         $condition
            //     GROUP BY
            //         customer_id,department_id,billDivName,currency
            //     ORDER BY
            //         customer_id
            // ");

            $reportData = DB::select("SELECT
                    customer_id AS customerID,
                    department_id,
                    billDivName,
                    currency,

                    SUM(CASE WHEN DATE_FORMAT(created_at, '%Y-%m') = DATE_FORMAT(LAST_DAY('$startDate'), '%Y-%m')
                            THEN CAST(due AS DECIMAL(18,2)) ELSE 0 END) AS `00 Month`,

                    SUM(CASE WHEN DATE_FORMAT(created_at, '%Y-%m') = DATE_FORMAT(DATE_SUB(LAST_DAY('$startDate'), INTERVAL 1 MONTH), '%Y-%m')
                            THEN CAST(due AS DECIMAL(18,2)) ELSE 0 END) AS `01 Month`,

                    SUM(CASE WHEN DATE_FORMAT(created_at, '%Y-%m') = DATE_FORMAT(DATE_SUB(LAST_DAY('$startDate'), INTERVAL 2 MONTH), '%Y-%m')
                            THEN CAST(due AS DECIMAL(18,2)) ELSE 0 END) AS `02 Month`,

                    SUM(CASE WHEN DATE_FORMAT(created_at, '%Y-%m') = DATE_FORMAT(DATE_SUB(LAST_DAY('$startDate'), INTERVAL 3 MONTH), '%Y-%m')
                            THEN CAST(due AS DECIMAL(18,2)) ELSE 0 END) AS `03 Month`,

                    -- 4–12 months ago
                    SUM(
                        CASE WHEN DATE_FORMAT(created_at, '%Y-%m') BETWEEN
                                DATE_FORMAT(DATE_SUB(LAST_DAY('$startDate'), INTERVAL 12 MONTH), '%Y-%m')
                            AND DATE_FORMAT(DATE_SUB(LAST_DAY('$startDate'), INTERVAL 4 MONTH), '%Y-%m')
                            THEN CAST(due AS DECIMAL(18,2)) ELSE 0 END
                    ) AS `>3 Month & <13`,

                    -- Older than 12 months
                    SUM(CASE WHEN created_at < DATE_SUB(LAST_DAY('$startDate'), INTERVAL 13 MONTH)
                            THEN CAST(due AS DECIMAL(18,2)) ELSE 0 END) AS `>12 Months`

                FROM bill_processes
                WHERE
                    created_at <= LAST_DAY('$startDate')
                    AND delete_flag = 0
                    $condition
                GROUP BY
                    customer_id, department_id, billDivName, currency
                ORDER BY
                    customer_id
            ");



            return view('report.customer-outstanding-ageing-bill-dept', compact('reportData', 'allCustomers'));
        }

        return view('report.customer-outstanding-ageing-bill-dept', compact('allCustomers'));
    }

    // Customer wise bill outstanding inspired from old billing software used in sil
    public function customerBillOutstanding(Request $request){

        $allCustomers = Customer::select('id','customer_name')
            ->orderBy('customer_name', 'asc')
            ->get();

        if ($request->has('start_date') && $request->has('customer_id')) {

            $start_date = $request->input('start_date');
            $customer_id = $request->input('customer_id');

            // $billProcesses = BillProcess::where('bill_processes.customer_id', $customer_id)
            //     ->where('bill_processes.delete_flag', 0)
            //     ->whereRaw("STR_TO_DATE(bill_processes.invoice_date, '%d/%m/%Y') <= STR_TO_DATE(?, '%d/%m/%Y')", [$start_date])
            //     ->leftJoin('collection_chds as c', 'bill_processes.invoice_no', '=', 'c.invoice_no')
            //     ->select(
            //         'bill_processes.customer_id',
            //         'bill_processes.invoice_no',
            //         'bill_processes.invoice_date',
            //         'bill_processes.bill_subject_desc',
            //         DB::raw('SUM(bill_processes.total_amount_vat) as total_amount_vat'),
            //         DB::raw('SUM(bill_processes.adjustment) as total_adjustment'),
            //         DB::raw('IFNULL(SUM(CAST(c.collection_amount as DECIMAL(15,2))),0) as total_collection_amount'),
            //         DB::raw('SUM(bill_processes.due) as total_due'),
            //     )
            //     ->with('customer:id,customer_name')
            //     ->groupBy('bill_processes.customer_id', 'bill_processes.invoice_no', 'bill_processes.invoice_date', 'bill_processes.bill_subject_desc')
            //     ->orderByRaw("STR_TO_DATE(bill_processes.invoice_date, '%d/%m/%Y') ASC") // 👈 Add this line
            //     ->get();

            $billProcesses = BillProcess::where('bill_processes.customer_id', $customer_id)
                ->where('bill_processes.delete_flag', 0)
                ->whereRaw("STR_TO_DATE(bill_processes.invoice_date, '%d/%m/%Y') <= STR_TO_DATE(?, '%d/%m/%Y')", [$start_date])
                ->leftJoin(DB::raw("
                    (
                        SELECT
                            invoice_no,
                            SUM(CAST(collection_amount AS DECIMAL(15,2))) AS total_collection_amount
                        FROM collection_chds
                        GROUP BY invoice_no
                    ) as c
                "), 'bill_processes.invoice_no', '=', 'c.invoice_no')
                ->select(
                    'bill_processes.customer_id',
                    'bill_processes.invoice_no',
                    'bill_processes.invoice_date',
                    'bill_processes.billDivName',
                    'bill_processes.bill_subject_desc',
                    DB::raw('SUM(bill_processes.total_amount_vat) as total_amount_vat'),
                    DB::raw('SUM(bill_processes.adjustment) as total_adjustment'),
                    DB::raw('IFNULL(c.total_collection_amount, 0) as total_collection_amount'),
                    DB::raw('SUM(bill_processes.due) as total_due'),
                )
                ->with('customer:id,customer_name')
                ->groupBy(
                    'bill_processes.customer_id',
                    'bill_processes.invoice_no',
                    'bill_processes.invoice_date',
                    'bill_processes.billDivName',
                    'bill_processes.bill_subject_desc',
                    'c.total_collection_amount'
                )
                ->orderByRaw("STR_TO_DATE(bill_processes.invoice_date, '%d/%m/%Y') ASC")
                ->get();

            // Format output
            $result = [];
            if ($billProcesses->isNotEmpty()) {
                $customerId = $billProcesses->first()->customer_id;
                $result[$customerId]['rows'] = $billProcesses;
                $result[$customerId]['total'] = [
                    'total_amount_vat'        => $billProcesses->sum('total_amount_vat'),
                    'total_adjustment'        => $billProcesses->sum('total_adjustment'),
                    'total_collection_amount' => $billProcesses->sum('total_collection_amount'),
                    'total_due'               => $billProcesses->sum('due'),
                ];
                $result[$customerId]['customer_name'] = $billProcesses->first()->customer->customer_name ?? '';
            }

            return view('report.customer-wise-bill-outstanding', compact('result', 'allCustomers'));
        }

        return view('report.customer-wise-bill-outstanding', compact('allCustomers'));
    }


    public function deptWiseMonthlySales(Request $request)
    {

        $allCustomers = Customer::select('id', 'customer_name')->get();

        if ($request->has('start_date') && $request->has('end_date')) {

            $startDate = Carbon::createFromFormat('d/m/Y', $request->input('start_date'))->format('Y-m-d');
            $endDate = Carbon::createFromFormat('d/m/Y', $request->input('end_date'))->format('Y-m-d');

            $condition = '';
            $start = Carbon::parse($startDate)->startOfMonth();  // Start from the beginning of the month
            $end = Carbon::parse($endDate)->endOfMonth(); // Get the end of the month of the end date

            $months = [];
            foreach (CarbonPeriod::create($start, '1 month', $end) as $date) {
                $months[] = $month_year = $date->format('F_Y');
                $monthName[] = $date->format('F');

                $monthNo = $date->format('n');
                $year = $date->format('Y');

                // $condition.= "SUM(CASE WHEN MONTH(created_at) = $monthNo AND YEAR(created_at) = $year THEN total_amount_vat ELSE 0 END) AS $month_year,";

                $condition.= "SUM(CASE WHEN MONTH(STR_TO_DATE(invoice_date, '%d/%m/%Y')) = $monthNo AND YEAR(STR_TO_DATE(invoice_date, '%d/%m/%Y')) = $year THEN total_amount ELSE 0 END) AS $month_year,";
            }

            $reportData = DB::select("SELECT
                    department_id,
                    $condition
                    billDivName
                FROM bill_processes
                WHERE
                delete_flag = 0 AND
                STR_TO_DATE(invoice_date, '%d/%m/%Y') BETWEEN '$startDate' AND '$endDate'
                GROUP BY department_id, billDivName
                ORDER BY department_id;
            ");

            return view('report.dept-wise-monthly-sales', compact('reportData','months','monthName'));
        }

        return view('report.dept-wise-monthly-sales');
    }

    public function deptWiseMonthlySalesOLDBACKUP(Request $request)
    {

        $allCustomers = Customer::select('id', 'customer_name')->get();

        if ($request->has('start_date') && $request->has('end_date')) {

            $startDate = Carbon::createFromFormat('d/m/Y', $request->input('start_date'))->format('Y-m-d');
            $endDate = Carbon::createFromFormat('d/m/Y', $request->input('end_date'))->format('Y-m-d');

            $condition = '';
            $start = Carbon::parse($startDate)->startOfMonth();  // Start from the beginning of the month
            $end = Carbon::parse($endDate)->endOfMonth(); // Get the end of the month of the end date

            $months = [];
            foreach (CarbonPeriod::create($start, '1 month', $end) as $date) {
                $months[] = $month_year = $date->format('F_Y');
                $monthName[] = $date->format('F');

                $monthNo = $date->format('n');
                $year = $date->format('Y');

                // $condition.= "SUM(CASE WHEN MONTH(created_at) = $monthNo AND YEAR(created_at) = $year THEN total_amount_vat ELSE 0 END) AS $month_year,";

                $condition.= "SUM(CASE WHEN MONTH(STR_TO_DATE(invoice_date, '%m/%d/%Y')) = $monthNo AND YEAR(STR_TO_DATE(invoice_date, '%m/%d/%Y')) = $year THEN total_amount_vat ELSE 0 END) AS $month_year,";
            }

            // return $condition;

//             $reportData = DB::select("SELECT department_id,
// SUM(
//   CASE
//     WHEN MONTH(STR_TO_DATE(invoice_date, '%m/%d/%Y')) = 5
//     AND YEAR(STR_TO_DATE(invoice_date, '%m/%d/%Y')) = 2024 THEN total_amount_vat
//     ELSE 0
//   END
// ) AS May_2024,
// SUM(
//   CASE
//     WHEN MONTH(STR_TO_DATE(invoice_date, '%m/%d/%Y')) = 6
//     AND YEAR(STR_TO_DATE(invoice_date, '%m/%d/%Y')) = 2024 THEN total_amount_vat
//     ELSE 0
//   END
// ) AS June_2024,
// SUM(
//   CASE
//     WHEN MONTH(STR_TO_DATE(invoice_date, '%m/%d/%Y')) = 7
//     AND YEAR(STR_TO_DATE(invoice_date, '%m/%d/%Y')) = 2024 THEN total_amount_vat
//     ELSE 0
//   END
// ) AS July_2024,
// SUM(
//   CASE
//     WHEN MONTH(STR_TO_DATE(invoice_date, '%m/%d/%Y')) = 8
//     AND YEAR(STR_TO_DATE(invoice_date, '%m/%d/%Y')) = 2024 THEN total_amount_vat
//     ELSE 0
//   END
// ) AS August_2024,
// SUM(
//   CASE
//     WHEN MONTH(STR_TO_DATE(invoice_date, '%m/%d/%Y')) = 9
//     AND YEAR(STR_TO_DATE(invoice_date, '%m/%d/%Y')) = 2024 THEN total_amount_vat
//     ELSE 0
//   END
// ) AS September_2024,
// SUM(
//   CASE
//     WHEN MONTH(STR_TO_DATE(invoice_date, '%m/%d/%Y')) = 10
//     AND YEAR(STR_TO_DATE(invoice_date, '%m/%d/%Y')) = 2024 THEN total_amount_vat
//     ELSE 0
//   END
// ) AS October_2024,
// SUM(
//   CASE
//     WHEN MONTH(STR_TO_DATE(invoice_date, '%m/%d/%Y')) = 11
//     AND YEAR(STR_TO_DATE(invoice_date, '%m/%d/%Y')) = 2024 THEN total_amount_vat
//     ELSE 0
//   END
// ) AS November_2024,
// SUM(
//   CASE
//     WHEN MONTH(STR_TO_DATE(invoice_date, '%m/%d/%Y')) = 12
//     AND YEAR(STR_TO_DATE(invoice_date, '%m/%d/%Y')) = 2024 THEN total_amount_vat
//     ELSE 0
//   END
// ) AS December_2024,
// billDivName
// FROM
//   bill_processes
// WHERE
//   STR_TO_DATE(invoice_date, '%m/%d/%Y') BETWEEN '2024-05-01' AND '2024-12-31'
// GROUP BY
//   department_id,
//   billDivName
// ORDER BY
//   department_id;
// ;");

            $reportData = DB::select("SELECT
                    department_id,
                    $condition
                    -- SUM(CASE WHEN MONTH(created_at) = 5 AND YEAR(created_at) = 2024 THEN total_amount_vat ELSE 0 END) AS May_2024,
                    -- SUM(CASE WHEN MONTH(created_at) = 6 AND YEAR(created_at) = 2024 THEN total_amount_vat ELSE 0 END) AS June_2024,
                    -- SUM(CASE WHEN MONTH(created_at) = 7 AND YEAR(created_at) = 2024 THEN total_amount_vat ELSE 0 END) AS July_2024,
                    -- SUM(CASE WHEN MONTH(created_at) = 8 AND YEAR(created_at) = 2024 THEN total_amount_vat ELSE 0 END) AS August_2024,
                    -- SUM(CASE WHEN MONTH(created_at) = 9 AND YEAR(created_at) = 2024 THEN total_amount_vat ELSE 0 END) AS September_2024,
                    -- SUM(CASE WHEN MONTH(created_at) = 10 AND YEAR(created_at) = 2024 THEN total_amount_vat ELSE 0 END) AS October_2024,
                    billDivName
                FROM bill_processes
                WHERE STR_TO_DATE(invoice_date, '%m/%d/%Y') BETWEEN '$startDate' AND '$endDate'
                GROUP BY department_id, billDivName
                ORDER BY department_id;
            ");

            // $reportData = DB::select("SELECT
            //         department_id,
            //         $condition
            //         -- SUM(CASE WHEN MONTH(created_at) = 5 AND YEAR(created_at) = 2024 THEN total_amount_vat ELSE 0 END) AS May_2024,
            //         -- SUM(CASE WHEN MONTH(created_at) = 6 AND YEAR(created_at) = 2024 THEN total_amount_vat ELSE 0 END) AS June_2024,
            //         -- SUM(CASE WHEN MONTH(created_at) = 7 AND YEAR(created_at) = 2024 THEN total_amount_vat ELSE 0 END) AS July_2024,
            //         -- SUM(CASE WHEN MONTH(created_at) = 8 AND YEAR(created_at) = 2024 THEN total_amount_vat ELSE 0 END) AS August_2024,
            //         -- SUM(CASE WHEN MONTH(created_at) = 9 AND YEAR(created_at) = 2024 THEN total_amount_vat ELSE 0 END) AS September_2024,
            //         -- SUM(CASE WHEN MONTH(created_at) = 10 AND YEAR(created_at) = 2024 THEN total_amount_vat ELSE 0 END) AS October_2024,
            //         billDivName
            //     FROM bill_processes
            //     WHERE created_at BETWEEN '$startDate' AND '$endDate  23:59:59'
            //     GROUP BY department_id, billDivName
            //     ORDER BY department_id;
            // ");

            return view('report.dept-wise-monthly-sales', compact('reportData','months','monthName'));
        }

        return view('report.dept-wise-monthly-sales');
    }












    // properly not working data not show(3)

    // public function customerOutstandingAgeingBill(Request $request)
    // {
    //     $allCustomers = Customer::select('id', 'customer_name')->get();

    //     if ($request->has('start_date')) {

    //         $startDate = Carbon::parse($request->input('start_date'));

    //         $pastMonths = [];
    //         for ($i = 0; $i < 12; $i++) {
    //             $pastMonths[] = [
    //                 'month' => $startDate->format('F'),
    //                 'year' => $startDate->format('Y'),
    //                 'day' => $startDate->format('d'),
    //                 'month_year' => $startDate->format('F Y'),
    //                 'full_date' => $startDate->format('Y-m-d')
    //             ];
    //             $startDate->subMonth();
    //         }
    //         $pastMonths = array_reverse($pastMonths);

    //         $billsQuery = BillProcess::where('delete_flag', 0);

    //         if ($request->filled('start_date')) {
    //             $billsQuery->whereDate('start_date', $request->input('start_date'));
    //         }

    //         $monthsYears = array_map(function ($monthYear) {
    //             return $monthYear['full_date'];
    //         }, $pastMonths);

    //         $billsQuery->whereIn(DB::raw("DATE_FORMAT(start_date, '%Y-%m-%d')"), $monthsYears)
    //             ->select('customer_id', DB::raw('SUM(total_amount_vat) as total_amount_vat'),
    //                      DB::raw("DATE_FORMAT(start_date, '%Y-%m-%d') as start_date"))
    //             ->groupBy('customer_id', 'start_date');

    //         if ($request->filled('customer_id')) {
    //             $selectedCustomerIds = $request->input('customer_id');
    //             $billsQuery->whereIn('customer_id', $selectedCustomerIds);

    //             $customers = Customer::whereIn('id', $selectedCustomerIds)->select('id', 'customer_name')->get();
    //         } else {
    //             $customers = Customer::select('id', 'customer_name')->get();
    //         }

    //         $bills = $billsQuery->get();

    //         $billsByCustomer = [];
    //         foreach ($bills as $bill) {
    //             $monthYearKey = Carbon::parse($bill->start_date)->format('Y-m-d');
    //             $billsByCustomer[$bill->customer_id][$monthYearKey] = $bill->total_amount_vat;
    //         }

    //         $reportData = [];
    //         foreach ($customers as $customer) {
    //             $data = [
    //                 'id' => $customer->id,
    //                 'name' => $customer->customer_name,
    //                 'bills' => []
    //             ];

    //             foreach ($pastMonths as $date) {
    //                 $monthYearKey = $date['day'] . ' ' . $date['month'] . ' ' . $date['year'];
    //                 $total_bill = $billsByCustomer[$customer->id][$monthYearKey] ?? 0;

    //                 $data['bills'][] = [
    //                     'total_bill' => $total_bill,
    //                     'month' => $date['month'],
    //                     'year' => $date['year'],
    //                     'day' => $date['day'],
    //                     'full_date' => $date['full_date']
    //                 ];
    //             }

    //             $reportData[] = $data;
    //         }

    //         return view('report.customer-outstanding-ageing-bill', compact('reportData', 'pastMonths', 'allCustomers'));
    //     }

    //     return view('report.customer-outstanding-ageing-bill', compact('allCustomers'));
    // }












    // Posting-Unposting bills
    public function postingUnpostingBill(Request $request)
    {

        $query = Bill::query();

        if ($request->filled('start_date')) {
            $startDate = Carbon::createFromFormat('m-Y', $this->normalizeDate($request->start_date))->startOfMonth()->format('Y-m-d');
            $query->where(DB::raw("STR_TO_DATE(created_at, '%Y-%m-%d')"), '>=', $startDate);
        }
        if ($request->filled('end_date')) {
            $endDate = Carbon::createFromFormat('m-Y', $this->normalizeDate($request->end_date))->endOfMonth()->format('Y-m-d');
            $query->where(DB::raw("STR_TO_DATE(created_at, '%Y-%m-%d')"), '<=', $endDate);
        }

        if ($request->filled('customer')) {
            $query->where('customer_id', $request->customer);
        }

        $query->where('status', 'Active');
        $query->where('approval', 1);
        $data = $query->get();

        $bills=[];

        foreach ($data as $value) {
            $processedBill=BillProcess::where('bill_id', $value->id)->first();

            $type = $request->type;
            if($type=='posted'){
                if($processedBill){
                    $bills[]=$value;
                }
            }else{
                if(!$processedBill){
                    $bills[]=$value;
                }
            }
        }

        $distinctCustomerIds = Bill::select('customer_id')->distinct()->pluck('customer_id');
        $customers = Customer::whereIn('id', $distinctCustomerIds)
            ->select('id','customer_name')
            ->orderBy('customer_name', 'asc')
            ->get();

        return view('report.posted-unposted-bill', compact('bills','customers'));

    }


    public function customerLedger(Request $request)
    {
        $openingBalance = 0;


        $ledgers = [];
        $customerName = '';

        if ($request->filled('start_date')) {
            $query = CustomerLedger::query();

            $startDate = Carbon::createFromFormat('d/m/Y', $this->normalizeDate($request->start_date))->format('Y-m-d');
            $query->where(DB::raw("STR_TO_DATE(entry_date, '%Y-%m-%d')"), '>=', $startDate);

            $queryOpening = CustomerLedger::query();
            $queryOpening->where(DB::raw("STR_TO_DATE(entry_date, '%Y-%m-%d')"), '<', $startDate);
            $queryOpening->where('customer_id', $request->customer);
            $ledgersOpening = $queryOpening->get();
            if($ledgersOpening){
                foreach ($ledgersOpening as $row) {
                    $openingBalance = $openingBalance + $row->amount;
                }
            }

        }

        if ($request->filled('end_date')) {
            $endDate = Carbon::createFromFormat('d/m/Y', $this->normalizeDate($request->end_date))->format('Y-m-d');
            $query->where(DB::raw("STR_TO_DATE(entry_date, '%Y-%m-%d')"), '<=', $endDate);
        }

        if ($request->filled('customer')) {
            $query->where('customer_id', $request->customer);
            $ledgers = $query->get();
            $customerInfo = Customer::find($request->customer);
            $customerName = $customerInfo->customer_name;
        }


        // $distinctCustomerIds = CustomerLedger::select('customer_id')->distinct()->pluck('customer_id');
        $customers = Customer::select('id','customer_name')
            ->orderBy('customer_name', 'asc')
            ->get();


        return view('report.customer-ledger', compact('customers', 'ledgers','openingBalance','customerName'));

    }


    // Process Bill List
    public function processBillReport(Request $request)
    {

        $perPage = $request->input('per_page', 25);

        $query = BillProcess::query();
        $query->where('delete_flag', 0);

        if ($request->filled('start_date')) {
            $startDate = Carbon::createFromFormat('d/m/Y', $this->normalizeDate($request->start_date))->format('Y-m-d');
            $query->whereDate('created_at', '>=', $startDate);
        }

        if ($request->filled('end_date')) {
            $endDate = Carbon::createFromFormat('d/m/Y', $this->normalizeDate($request->end_date))->format('Y-m-d');
            $query->whereDate('created_at', '<=', $endDate);
        }

        if ($request->filled('process_no_format')) {
            $query->where('process_no_format', $request->process_no_format);
        }

        // $groupedBills = $query->select(
        //     'process_no_format'
        // )
        // ->groupBy('process_no_format')
        // ->orderBy('created_at', 'desc')
        // ->paginate($perPage)
        // ->appends($request->except('page'));

        $groupedBills = $query->select(
            'process_no_format',
            DB::raw('MAX(created_at) as recent_created_at') // Add MAX(created_at)
        )
        ->groupBy('process_no_format')
        ->orderByDesc('recent_created_at') // Order by the aggregated column
        ->paginate($perPage)
        ->appends($request->except('page'));

        return view('invoice.process-bill-report', compact('groupedBills', 'perPage'));

    }

    // Process bill preview
    public function processBillReportView($processNo)
    {
        $processNo = str_replace("_","/",$processNo);

        $processInfo = BillProcess::select('month','year')->where('process_no_format', $processNo)->first();

        $billProcesses = BillProcess::where('process_no_format', $processNo)
        ->select(
            'customer_id',
            'invoice_no',
            'invoice_date',
            'bill_subject_desc',
            'billDivName',
            DB::raw('SUM(total_amount) as total_amount'),
            DB::raw('SUM(vat_amount) as vat_amount'),
            DB::raw('SUM(total_amount_vat) as total_amount_vat')
        )
        ->with(['billSubject', 'customer']) // Load related Customer data
        ->groupBy('customer_id', 'invoice_no', 'invoice_date', 'bill_subject_desc', 'billDivName')
        ->orderBy('customer_id')
        ->get()
        ->groupBy('customer_id');

        $result = [];
        foreach ($billProcesses as $customerId => $bills) {
            $result[$customerId]['rows'] = $bills;
            $result[$customerId]['total'] = [
                'total_amount' => $bills->sum('total_amount'),
                'vat_amount' => $bills->sum('vat_amount'),
                'total_amount_vat' => $bills->sum('total_amount_vat'),
            ];
            $result[$customerId]['customer_name'] = $bills->first()->customer->customer_name ?? '';
        }

        return view('invoice.process-bill-report-view', compact('result','processNo','processInfo'));

    }

    // Posted/Processed bill list Pdf generate
    public function processBillReportPdf($processNo){

        $data['processNo'] = $processNo = str_replace("_","/",$processNo);

        $data['processInfo'] = $pInfo = BillProcess::select('month','year')->where('process_no_format', $processNo)->first();

        $billProcesses = BillProcess::where('process_no_format', $processNo)
        ->select(
            'customer_id',
            'invoice_no',
            'invoice_date',
            'bill_subject_desc',
            'billDivName',
            DB::raw('SUM(total_amount) as total_amount'),
            DB::raw('SUM(vat_amount) as vat_amount'),
            DB::raw('SUM(total_amount_vat) as total_amount_vat')
        )
        ->with(['billSubject', 'customer']) // Load related Customer data
        ->groupBy('customer_id', 'invoice_no', 'invoice_date', 'bill_subject_desc', 'billDivName')
        ->orderBy('customer_id')
        ->get()
        ->groupBy('customer_id');

        $result = [];
        foreach ($billProcesses as $customerId => $bills) {
            $result[$customerId]['rows'] = $bills;
            $result[$customerId]['total'] = [
                'total_amount' => $bills->sum('total_amount'),
                'vat_amount' => $bills->sum('vat_amount'),
                'total_amount_vat' => $bills->sum('total_amount_vat'),
            ];
            $result[$customerId]['customer_name'] = $bills->first()->customer->customer_name ?? 'N/A';
        }

        $data['result'] = $result;
        $name = "Process_bill_report_for_".$pInfo->month."_".$pInfo->year.".pdf";
        $pdf = Pdf::loadView('invoice.process-bill-report-pdf', $data);
        $pdf->setPaper('A4', 'landscape');
        // return $pdf->stream($name, ['Attachment'=>false]);
        return $pdf->download($name); // Force download

    }

    public function showBillView($id){

        $bill = Bill::find($id);
        return view('bill-process.showBillView', compact('bill'));
    }

    public function billRevisionHistoryTest(Request $request)
    {
        if (!auth()->user()->can('View Report')) {
            abort(403, 'Sorry !! You are not authorized!');
        }

        $bills = [];
        $query = Bill::query();

        $query->select([
            'billing_bills.id',
            'customer_id',
            'billDivName',
            'bill_subject_id',
            'particulars',
            'ref_no',
            'ref_date',
            'qty',
            'unit_price_vat',
            'vat_status',
            'vat_percent',
            'total_amount',
            'vat_amount',
            'total_amount_vat',
            'installation_date',
            'bill_type',
            'start_date',
            'yearly_increase_percent',
            'increase_start_date',
            'end_date',
            'bill_group',
            'billing_customers.status',
            'billing_customers.customer_name',
        ]);

        $query->join('billing_customers', 'billing_bills.customer_id', '=', 'billing_customers.id');
        $query->where('billing_customers.status', 'Active');
        $query->where('billing_bills.approval', 1);
        $query->where('billing_bills.delete_flag', 0);

        if ($request->filled('customer')) {
            $query->where('customer_id', $request->customer);
            $query->orderBy('billing_bills.id', 'asc');
        }else{
            $query->orderBy('billing_customers.customer_name', 'asc')
                ->orderBy('billing_bills.id', 'asc');
        }

        if ($request->filled('bill_no')) {
            $query->where('billing_bills.id', $request->bill_no);
        }

        if ($request->filled('status')) {
            $query->where('billing_bills.status', $request->status);
        }

        if ($request->query->count() > 0) {
            $query->with(['billSubject:id,bill_subject']);
            $bills = $query->get();
        }

        $customers = Customer::select('id','customer_name')
            ->orderBy('customer_name', 'asc')
            ->get();

        return view('report.bill-revision-history', compact('bills','customers'));
    }

    public function billRevisionHistory(Request $request)
    {
        if (!auth()->user()->can('View Report')) {
            abort(403, 'Sorry !! You are not authorized!');
        }

        $bills = collect(); // default empty
        $query = BillRevision::query();

        $query->select([
            'billing_bill_revisions.id',
            'billing_bill_revisions.bill_id',
            'billing_bill_revisions.revision_no',
            'billing_bill_revisions.customer_id',
            'billing_bill_revisions.billDivName',
            'billing_bill_revisions.bill_subject_id',
            'billing_bill_revisions.particulars',
            'billing_bill_revisions.ref_no',
            'billing_bill_revisions.ref_date',
            'billing_bill_revisions.qty',
            'billing_bill_revisions.unit_price_vat',
            'billing_bill_revisions.vat_status',
            'billing_bill_revisions.vat_percent',
            'billing_bill_revisions.total_amount',
            'billing_bill_revisions.vat_amount',
            'billing_bill_revisions.total_amount_vat',
            'billing_bill_revisions.installation_date',
            'billing_bill_revisions.bill_type',
            'billing_bill_revisions.start_date',
            'billing_bill_revisions.yearly_increase_percent',
            'billing_bill_revisions.increase_start_date',
            'billing_bill_revisions.end_date',
            'billing_bill_revisions.bill_group',
            'billing_bill_revisions.revision_no',
            'billing_customers.status',
            'billing_customers.customer_name',
        ]);

        $query->join('billing_customers', 'billing_bill_revisions.customer_id', '=', 'billing_customers.id');
        $query->where('billing_customers.status', 'Active');
        $query->where('billing_bill_revisions.delete_flag', 0);

        // Filter by Bill No (required)
        if ($request->filled('bill_no')) {
            $query->where('billing_bill_revisions.bill_id', $request->bill_no);
            $query->orderBy('billing_bill_revisions.revision_no', 'asc');
        }

        // Optional filters
        if ($request->filled('customer')) {
            $query->where('billing_bill_revisions.customer_id', $request->customer);
        }

        if ($request->filled('status')) {
            $query->where('billing_bill_revisions.status', $request->status);
        }

        if ($request->query->count() > 0) {
            $query->with(['billSubject:id,bill_subject']);
            $bills = $query->get();
        }

        $customers = Customer::select('id', 'customer_name')
            ->orderBy('customer_name', 'asc')
            ->get();

        return view('report.bill-revision-history', compact('bills', 'customers'));
    }





    public function billIncreaseWarningReport(Request $request)
    {
        if (!auth()->user()->can('View Report')) {
            abort(403, 'Sorry !! You are not authorized!');
        }

        $bills = [];
        $query = DB::table('billing_bills as b');

        $query->select([
            'b.ref_no',
            'b.ref_date',
            'c.customer_name',
            'b.particulars',
            'b.start_date',
            'b.end_date',
            'b.increase_past_date',
            'b.increase_start_date',
            'b.yearly_increase_percent',
            'b.increase_period',
            'b.bill_no_x as OLD_BILL_NO',
            'b.bill_group',
            'b.id as BILL_NO',
        ]);

        $query->leftJoin('billing_customers as c', 'b.customer_id', '=', 'c.id');

        $query->where('c.status', 'Active');
        $query->where('b.approval', 1);
        $query->where('b.delete_flag', 0);

        if ($request->filled('customer')) {
            $query->where('b.customer_id', $request->customer);
            $query->orderBy('b.id', 'asc');
        } else {
            $query->orderBy('c.customer_name', 'asc')
                ->orderBy('b.id', 'asc');
        }

        if ($request->filled('start_date')) {
            // b.increase_period MUST be > 0 (ignore null/blank/"0")
            $query->whereRaw("CAST(NULLIF(b.increase_period, '') AS UNSIGNED) > 0");
            $query->whereRaw(
                "STR_TO_DATE(b.increase_start_date, '%d/%m/%Y') <= STR_TO_DATE(?, '%d/%m/%Y')",
                [$request->start_date]
            );
            $bills = $query->get();
        }

        $customers = Customer::select('id','customer_name')
            ->orderBy('customer_name', 'asc')
            ->get();

        return view('report.bill-increase-warning-report', compact('bills','customers'));
    }

    public function billIncreaseWarningReportExport(Request $request, $type)
    {
        $bills = [];
        $query = DB::table('billing_bills as b');

        $query->select([
            'b.ref_no',
            'b.ref_date',
            'c.customer_name',
            'b.particulars',
            'b.start_date',
            'b.end_date',
            'b.increase_past_date',
            'b.increase_start_date',
            'b.yearly_increase_percent',
            'b.increase_period',
            'b.bill_no_x as OLD_BILL_NO',
            'b.bill_group',
            'b.id as BILL_NO',
        ]);

        $query->leftJoin('billing_customers as c', 'b.customer_id', '=', 'c.id');

        $query->where('c.status', 'Active');
        $query->where('b.approval', 1);
        $query->where('b.delete_flag', 0);

        if ($request->filled('rpt_customer')) {
            $query->where('b.customer_id', $request->rpt_customer);
            $query->orderBy('b.id', 'asc');
        } else {
            $query->orderBy('c.customer_name', 'asc')
                ->orderBy('b.id', 'asc');
        }

        if ($request->filled('rpt_start_date')) {
            // b.increase_period MUST be > 0 (ignore null/blank/"0")
            $query->whereRaw("CAST(NULLIF(b.increase_period, '') AS UNSIGNED) > 0");
            $query->whereRaw(
                "STR_TO_DATE(b.increase_start_date, '%d/%m/%Y') <= STR_TO_DATE(?, '%d/%m/%Y')",
                [$request->rpt_start_date]
            );
            $bills = $query->get();
        }

        $data['bills'] = $bills;
        $data['customer'] = $request->rpt_customer;
        $data['start_date'] = $request->rpt_start_date;

        if ($type == 'pdf') {
            $name = "Bill_Increase_Warning_Report_" . date('Y-m-d') . ".pdf";
            $pdf = Pdf::loadView('report.pdf.bill-increase-warning-report-pdf', $data);
            $pdf->setPaper('Legal', 'landscape');
            // return $pdf->stream($name, ['Attachment'=>false]);
            return $pdf->download($name); // Force download
        }

        // Uncomment when Excel export needed
        if ($type == 'excel') {
            $name = "Bill_Increase_Warning_Report_" . date('Y-m-d') . ".xlsx";
            return Excel::download(new BillIncreaseWarningExport($request), $name);
        }

        return back()->with('error', 'Invalid export type.');
    }

    public function billIncreaseWarningCount($date) // 27-05-2025
    {
        $count = DB::table('billing_bills as b')
            ->where('b.approval', 1)
            ->where('b.delete_flag', 0)
            ->whereRaw("CAST(NULLIF(b.increase_period, '') AS UNSIGNED) > 0")
            ->whereRaw(
                "STR_TO_DATE(b.increase_start_date, '%d/%m/%Y') <= STR_TO_DATE(?, '%d-%m-%Y')",
                [$date]
            )
            ->count();

        return $count;
    }



    // public function billIncreaseWarningReportDirect(Request $request)
    // {
    //     if (!auth()->user()->can('View Report')) {
    //         abort(403, 'Sorry !! You are not authorized!');
    //     }

    //     $bills = [];
    //     $query = DB::table('increase_report_sil_333 as i');

    //     $query->select([
    //         'b.ref_no',
    //         'b.ref_date',
    //         'c.customer_name',
    //         'b.particulars',
    //         'b.start_date',
    //         'b.end_date',
    //         'i.INCREASE_PAST_DATE',
    //         'i.INCREASE_DATE',
    //         'i.INCREASE_PERCENTAGE',
    //         'i.INCREASE_PERIOD',
    //         'i.BILL_NO',
    //         'b.bill_group',
    //         'b.id',
    //     ]);

    //     $query->leftJoin('billing_bills as b', 'i.BILL_NO', '=', 'b.bill_no_x');
    //     $query->leftJoin('billing_customers as c', 'b.customer_id', '=', 'c.id');

    //     $query->where('c.status', 'Active');
    //     $query->where('b.approval', 1);
    //     $query->where('b.delete_flag', 0);

    //     if ($request->filled('customer')) {
    //         $query->where('b.customer_id', $request->customer);
    //         $query->orderBy('b.id', 'asc');
    //     } else {
    //         $query->orderBy('c.customer_name', 'asc')
    //             ->orderBy('b.id', 'asc');
    //     }

    //     if ($request->filled('start_date')) {
    //         $query->whereRaw(
    //             "STR_TO_DATE(i.INCREASE_DATE, '%d/%m/%Y') <= STR_TO_DATE(?, '%d/%m/%Y')",
    //             [$request->start_date]
    //         );
    //         $bills = $query->get();
    //     }

    //     $customers = Customer::select('id','customer_name')
    //         ->orderBy('customer_name', 'asc')
    //         ->get();

    //     return view('report.bill-increase-warning-report', compact('bills','customers'));
    // }

    // public function billIncreaseWarningReportExport89890(Request $request, $type)
    // {

    //     $bills = [];
    //     $query = Bill::query();

    //     $query->select([
    //         'customer_id',
    //         'billDivName',
    //         'bill_subject_id',
    //         'particulars',
    //         'ref_no',
    //         'ref_date',
    //         'qty',
    //         'unit_price_vat',
    //         'vat_status',
    //         'vat_percent',
    //         'installation_date',
    //         'bill_type',
    //         'start_date',
    //         'yearly_increase_percent',
    //         'increase_start_date',
    //         'end_date',
    //         'bill_group',
    //         'billing_customers.status',
    //         'billing_customers.customer_name',
    //     ]);

    //     $query->join('billing_customers', 'billing_bills.customer_id', '=', 'billing_customers.id');
    //     $query->where('billing_customers.status', 'Active');
    //     $query->where('billing_bills.approval', 1);
    //     $query->where('billing_bills.delete_flag', 0);

    //     if ($request->filled('rpt_customer')) {
    //         $query->where('customer_id', $request->rpt_customer);
    //         $query->orderBy('billing_bills.id', 'asc');
    //     }else{
    //         $query->orderBy('billing_customers.customer_name', 'asc')
    //             ->orderBy('billing_bills.id', 'asc');
    //     }

    //     if ($request->filled('rpt_status')) {
    //         $query->where('billing_bills.status', $request->rpt_status);
    //         $query->with(['billSubject:id,bill_subject']);
    //         $bills = $query->get();
    //     }

    //     $data['bills'] = $bills;
    //     $data['customer'] = $request->rpt_customer;
    //     $data['status'] = $request->rpt_status;

    //     if($type=='pdf'){
    //         $name = "Billing_Particulars_for_Customer_".date('Y-m-d').".pdf";
    //         $pdf = Pdf::loadView('report.pdf.bill-increase-warning-report-pdf', $data);
    //         $pdf->setPaper('Legal', 'landscape');
    //         // return $pdf->stream($name, ['Attachment'=>false]);
    //         return $pdf->download($name); // Force download
    //     }

    //     // if($type=='excel'){
    //     //     $name = "Billing_Particulars_for_Customer_".date('Y-m-d').".xlsx";
    //     //     return Excel::download(new CustomerBillParticularExport($request), $name);
    //     // }

    // }

    // public function billIncreaseWarningReportExportDirect(Request $request, $type)
    // {
    //     $bills = [];
    //     $query = DB::table('increase_report_sil_333 as i');

    //     $query->select([
    //         'b.ref_no',
    //         'b.ref_date',
    //         'c.customer_name',
    //         'b.particulars',
    //         'b.start_date',
    //         'b.end_date',
    //         'i.INCREASE_PAST_DATE',
    //         'i.INCREASE_DATE',
    //         'i.INCREASE_PERCENTAGE',
    //         'i.INCREASE_PERIOD',
    //         'i.BILL_NO',
    //         'b.bill_group',
    //         'b.id',
    //     ]);

    //     $query->leftJoin('billing_bills as b', 'i.BILL_NO', '=', 'b.bill_no_x');
    //     $query->leftJoin('billing_customers as c', 'b.customer_id', '=', 'c.id');

    //     $query->where('c.status', 'Active');
    //     $query->where('b.approval', 1);
    //     $query->where('b.delete_flag', 0);

    //     if ($request->filled('rpt_customer')) {
    //         $query->where('b.customer_id', $request->rpt_customer);
    //         $query->orderBy('b.id', 'asc');
    //     } else {
    //         $query->orderBy('c.customer_name', 'asc')
    //             ->orderBy('b.id', 'asc');
    //     }

    //     if ($request->filled('rpt_start_date')) {
    //         $query->whereRaw(
    //             "STR_TO_DATE(i.INCREASE_DATE, '%d/%m/%Y') <= STR_TO_DATE(?, '%d/%m/%Y')",
    //             [$request->rpt_start_date]
    //         );
    //         $bills = $query->get();
    //     }

    //     $data['bills'] = $bills;
    //     $data['customer'] = $request->rpt_customer;
    //     $data['start_date'] = $request->rpt_start_date;

    //     if ($type == 'pdf') {
    //         $name = "Bill_Increase_Warning_Report_" . date('Y-m-d') . ".pdf";
    //         $pdf = Pdf::loadView('report.pdf.bill-increase-warning-report-pdf', $data);
    //         $pdf->setPaper('Legal', 'landscape');
    //         // return $pdf->stream($name, ['Attachment'=>false]);
    //         return $pdf->download($name); // Force download
    //     }

    //     // Uncomment when Excel export needed
    //     if ($type == 'excel') {
    //         $name = "Bill_Increase_Warning_Report_" . date('Y-m-d') . ".xlsx";
    //         return Excel::download(new BillIncreaseWarningExport($request), $name);
    //     }

    //     return back()->with('error', 'Invalid export type.');
    // }






}
